/* IC9700 Band decoder and PTT steering test */

/* Minimal circuit
CIV on D2 (RX) and D7 (TX)
Send input D6, active low
Active low outputs D3,4,5
Active high outputs D8,9,10
IC9700 at 9600 Baud CIV address 0xA2
D13 (the one with the LED fitted) is used to indicate CIV polling
*/
 
#include <SoftwareSerial.h>     // For CIV
#define RX_PIN 2                // Software serial pins
#define TX_PIN 7                // Software serial pins
#define radio_address 0xA2      // IC9700
#define controller_address 0x0  // Universal for all ICOM radios
#define civ_baudrate 9600       // CIV baud rate
#define monitor_baudrate 57600  // For serial monitoring
#define LS1 3                   // PTT output 1
#define LS2 4                   // PTT output 2
#define LS3 5                   // PTT output 3
#define HS1 8                   // Band output 1
#define HS2 9                   // Band output 2
#define HS3 10                  // Band output 3
#define PTT 6                   // PTT input
#define BOUNCE 10               // Number of loops before releasing PTT, stops chatter
#define TIMEOUT 10              // Timeout for CIV connections

unsigned char buf[65]; 
char currentband;
int CIVtimeout = 0;
int n,len;
char band[3];
volatile bool PTTstate = false;
bool debug = false;
volatile int bounce = BOUNCE; // This is the number of loops the program will run before releasing PTT, it is to stop chatter

unsigned long lasttime = 0;
const unsigned long polltime = 1000; // How long between band polls in mS

// Set up the software serial port
SoftwareSerial ic9700_Serial(RX_PIN, TX_PIN); // RX, TX

void setup() {

  pinMode(RX_PIN, INPUT);  // CI-V serial communication
  pinMode(TX_PIN, OUTPUT); // CI-V serial communication
  
  pinMode(LS1, OUTPUT); // e.g. 2m
  pinMode(LS2, OUTPUT); // e.g. 70cm
  pinMode(LS3, OUTPUT); // e.g. 23cm
  pinMode(HS1, OUTPUT); // e.g. 2m
  pinMode(HS2, OUTPUT); // e.g. 70cm
  pinMode(HS3, OUTPUT); // e.g. 23cm
  pinMode(PTT, INPUT); // e.g. 23cm

  // Start up with all off
  digitalWrite(LS1,LOW);
  digitalWrite(LS2,LOW);
  digitalWrite(LS3,LOW);
  digitalWrite(HS1,LOW);
  digitalWrite(HS2,LOW);
  digitalWrite(HS3,LOW);

  pinMode(13, OUTPUT); 
  digitalWrite(13, LOW); // force LED (pin 13) to turn off.
  
  // Initialise variables
  currentband = 0; // This is not a valid band so safe
  band[0] = 0;
  band[1] = 0;
  band[2] = 0;
  
  Serial.begin(monitor_baudrate); 
  
  Serial.println("IC9700 PTT and band select");
  
  ic9700_Serial.begin(civ_baudrate);
  ic9700_Serial.setTimeout(200);

  // PTT interrupt - falling edge enable PTT straight away.
  attachInterrupt(digitalPinToInterrupt(PTT), pttpressed, FALLING);
}
//-------------------------------------------------------------------------
bool checkbandchange()
  {
  // Checks for band changes periodically sending command to read main vfo band
  // returns true if band changed 
  
  unsigned char getfreq[]={0xFE,0xFE,radio_address,controller_address,0x25,0x00,0xFD};
  
  // Serial.println("Getting band ");
  //ic9700_Serial.flush();
  digitalWrite(13, HIGH);
  ic9700_Serial.write((char*)getfreq,7);

  //read response from radio
  len = ic9700_Serial.readBytesUntil(0xFD, buf, 64); 
  
  CIVtimeout--; // Decriment the CIV timeout timer 
     
  if(debug)
    {
    Serial.print(len); 
    Serial.println(" bytes received");
    //Serial.println(buf[10]);
    }
  digitalWrite(13, LOW);
  
  // Check it is what we wanted
  if(len != 11) // Right number of byted
    return false;

  if(buf[0]!=0xFE)  // Packet start alignment
    return false;  
    
  // If here looks like we have a valid return, reset timeout  
  CIVtimeout = TIMEOUT;
  
  if (buf[10]!=currentband) // Band change detected
    {          
    currentband=buf[10];             
    switch(buf[10]) // Reset the band to current
      {
      case 0x01:     
        Serial.println("Selected 2m");
        break;               
      case 0x04:
        Serial.println("Selected 70cm");
        break;
      case 0x12:
        Serial.println("Selected 23cm");
        break;
      }
    return true;   
    }    
  return false;
  }
//-------------------------------------------------------------------------
void changeband(char newband)
  {
  // If the band has changed, do things   
  switch(newband)
    {
    case 0x01:
      band[0]=1;
      band[1]=0;
      band[2]=0;          
      break;
        
    case 0x04:
      band[0]=0;
      band[1]=1;
      band[2]=0;          
      break;
        
    case 0x12:
      band[0]=0;
      band[1]=0;
      band[2]=1;          
      break;
    }        
  return;  
  }
//-------------------------------------------------------------------------
void pttpressed()
{
  // If the PTT pin changes to on we are here - instantly enact PTT
  // Note this never releases PTT (unless the band changed), that is done 
  // elsewhere 
  PTTstate = true;  
  digitalWrite(LS1,band[0]);
  digitalWrite(LS2,band[1]);
  digitalWrite(LS3,band[2]);
  bounce = BOUNCE;     
}

//-------------------------------------------------------------------------
void loop() {
  // Check for band chage periodically
  delay(1); // Simply to slow down this loop to about 1mS for PTT debouncing
  if ( millis() - lasttime > polltime)
   {
   lasttime = millis();    
   if(checkbandchange() == true)
     {
     changeband(currentband);         
     // Set band outputs 
     digitalWrite(HS1,band[0]);
     digitalWrite(HS2,band[1]);
     digitalWrite(HS3,band[2]);  
     }
   } 
   // If CIV timing lost disable everything
   if(CIVtimeout <= 0)
     {
      CIVtimeout = 0;
      currentband = 0;
      band[0] = 0;
      band[1] = 0;
      band[2] = 0;  
      digitalWrite(LS1,LOW);
      digitalWrite(LS2,LOW);
      digitalWrite(LS3,LOW);
      digitalWrite(HS1,LOW);
      digitalWrite(HS2,LOW);
      digitalWrite(HS3,LOW);        
      }
   // PTT control - read PTT line set accordingly
   // Note PTT is set by an interrupt for speed, this will simply verify 
   // this and if the PTT is unset it will go back to RX mode
   // but only after a defined set of itterations to prevent chatter
   // if the PTT switch bounces
   
   PTTstate = !digitalRead(PTT);
   if(PTTstate == true)
      {
      digitalWrite(LS1,band[0]);
      digitalWrite(LS2,band[1]);
      digitalWrite(LS3,band[2]);
      }
    else
      {
      bounce--;
      if(bounce <=0)
      {  
        digitalWrite(LS1,0);
        digitalWrite(LS2,0);
        digitalWrite(LS3,0);
      }
    }         
}
//-----------------------------------------------------------------------
